<?php

if (!defined('ABSPATH')) {
  exit; // Exit if accessed directly.
}

class Ultimate_AI_Orders
{
  private $history_manager, $USER_MANAGER, $UAI_SUBSCRIPTION;
  public function __construct()
  {
    $this->history_manager = new Ultimate_AI_History_Manager();
    $this->USER_MANAGER = new Ultimate_AI_User_Manager();
    $this->UAI_SUBSCRIPTION = new Ultimate_AI_Subscription();
  }

  // Users functions
  /** 
   * 
   * @since v1.0.0
   * Creating an order while creating subscription
   * Accepts subscription $data
   * @return $order_id
   * 
   */
  public function createOrder($data)
  {
    $post = [
      'post_author' => $data['user_id'],
      'post_title' => $data['name'],
      'post_status' => 'private',
      'post_type' => 'ultimate_ai_orders',
      'meta_input' => array(
        'customer_name'     => $data['name'],
        'customer_email'    => $data['email'],
        'user_id'           => $data['user_id'],
        'order_id'          => $data['order_id'],
        'subscription_id'   => isset($data['subscription_id']) ? $data['subscription_id'] : $data['order_id'],
        'starts'            => $data['starts'],
        'expiry'            => $data['expiry'],
        'price'             => $data['plan_price'],
        'pricing_id'        => $data['pricing_id'],
        'payment_status'    => $data['payment_status'],
        'payment_method'    => $data['payment_method'],
        'subscribed'        => $data['subscribed'],
        'country'           => $data['country'],
        'address'           => $data['address'],
        'currency'          => $data['currency'],
        'package_type'      => $data['package_type'],
        'package_name'      => $data['package_name'],
        'plan_interval'     => $data['package_name'],
        'plan_id'           => $data['plan_id'],
      )
    ];
    $order_id = wp_insert_post($post);

    update_post_meta($order_id, 'ultimate_ai_order_meta', $data);
    return $order_id;
  }

  public function updateOrderStatus($status, $order_id)
  {
    $order_meta = get_post_meta($order_id, 'ultimate_ai_order_meta', true);
    $order_meta['payment_status'] = $status;

    if ($status == 'hold' || $status == 'ban' || $status == 'cancel') {
      $order_meta['subscribed'] = false;
    }

    if ($status == 'cancel') {
      $order_meta['order_post_id'] = $order_id;
      $order_meta['subscribed'] = false;
      $order_meta['expiry'] = time();
      $order_meta['payment_status'] = 'cancel';
      $cancelled = $this->UAI_SUBSCRIPTION->updateSubcriptionStatus('CANCEL', $order_meta);
    }

    update_post_meta($order_id, 'payment_status', $status);
    update_post_meta($order_id, 'subscribed', $order_meta['subscribed']);
    return update_post_meta($order_id, 'ultimate_ai_order_meta', $order_meta);
  }

  // SALES($) / REVENUE - from TOKENS, SUBSCRIPTIONS, Todays received, ALL TIME, this week, this month, last week, last month   
  // COUNT - from TOKENS, SUBSCRIPTIONS, Todays, ALL TIME, this week, this month, last week, last month 
  // Generation Details - from Images, Todays, ALL TIME, this week, this month, last week, last month 
  public function UltimateAiRevenueData()
  {
    return [
      'orders' => $this->UltimateAiOrdersData('MONTH'),
      'earnings_today' => $this->UltimateAiEarnings('DAY'),
      'earnings' => $this->UltimateAiEarnings('MONTH')
    ];
  }

  public function UltimateAiUsersData()
  {
    $dates = $this->dateRangeCreator("MONTH");

    $change_pct = 0;
    $users_live = 0;
    // TOTAL MEMBERS
    $users_all = $this->USER_MANAGER->filter_users();

    // USERS IN THIS MONTH
    $now_args = [
      'date_query' => [
        array(
          'after'     => $dates['now_start'],
          'before'     => $dates['now_end'],
          'inclusive' => true,
        ),
      ]
    ];
    $users_current = $this->USER_MANAGER->filter_users($now_args);

    // USERS IN PREVIOUS MONTH
    $prev_args = [
      'date_query' => [
        array(
          'after'     => $dates['prev_start'],
          'before'     => $dates['prev_end'],
          'inclusive' => true,
        ),
      ]
    ];
    $users_prev = $this->USER_MANAGER->filter_users($prev_args);

    // ACTIVE USERS
    $active_args = [
      'meta_key' => 'session_tokens',
      'meta_compare' => 'EXISTS'
    ];
    $users_live = $this->USER_MANAGER->filter_users($active_args);

    $change_pct = (intval($users_current['users_count']) - intval($users_prev['users_count'])) / 100;
    $deflection = $change_pct < 0 ? 'NEG' : 'POS';

    return [
      'all' => $users_all['users_count'],
      'change' => $change_pct,
      'deflection' => $deflection,
      'live' => $users_live['users_count'],
    ];
  }
  public function UltimateAiOrdersData($range)
  {
    $dates = $this->dateRangeCreator($range);
    $orders_current = $this->filterOrders([
      'start' => $dates['now_start'],
      'end' => $dates['now_end'],
    ]);
    $orders_prev = $this->filterOrders([
      'start' => $dates['prev_start'],
      'end' => $dates['prev_end'],
    ]);

    $total_orders_current = 0;
    $total_orders_prev = 0;
    $change_text = "From last " . strtolower($range);
    foreach ($orders_current['all_orders'] as $order) {
      $total_orders_current += 1;
    }
    foreach ($orders_prev['all_orders'] as $order) {
      $total_orders_prev += 1;
    }

    $change_pct = ($total_orders_current - $total_orders_prev) / 100;

    return [
      'total' => $total_orders_current,
      'prev' => $total_orders_prev,
      'change' => $change_pct,
      'deflection' => $change_pct > 0 ? 'POS' : 'NEG',
      'text' => $change_text,
    ];
  }
  public function UltimateAiEarnings($range)
  {
    $dates = $this->dateRangeCreator($range);
    $orders_current = $this->filterOrders([
      'start' => $dates['now_start'],
      'end' => $dates['now_end'],
    ]);
    $orders_prev = $this->filterOrders([
      'start' => $dates['prev_start'],
      'end' => $dates['prev_end'],
    ]);

    $total_earnings_current = 0;
    $total_earnings_prev = 0;
    $change_text = "From last " . strtolower($range);
    foreach ($orders_current['all_orders'] as $order) {
      $total_earnings_current = $total_earnings_current + (int) $order['plan_price'];
    }
    foreach ($orders_prev['all_orders'] as $order) {
      $total_earnings_prev = $total_earnings_prev + (int) $order['plan_price'];
    }

    $change_pct = ($total_earnings_current - $total_earnings_prev) / 100;

    return [
      'total' => $total_earnings_current,
      'prev' => $total_earnings_prev,
      'change' => $change_pct,
      'deflection' => $change_pct > 0 ? 'POS' : 'NEG',
      'text' => $change_text,
    ];
  }

  /** This will create a date range for a time range
   *   For example you give a WEEK as range: 
   *       It will create date range last 7 days from today and previous week ending on the first day of the range date. 
   *       i.e: ['01-01-24', '01-07-24'] WHERE Today is 01-07-24
   *   If you give MONTH as $range param:
   *       It will create date range of this month and previous month
   *   @param String $range // DAY, WEEK, MONTH, YEAR
   *   @return Array [
   *        'now_end' => $now_end,
   *        'now_start' => $now_start,
   *        'prev_end' => $prev_end,
   *        'prev_start' => $prev_start,
   *           ]
   *
   */
  public function dateRangeCreator($range)
  {
    switch ($range) {
      case 'DAY':
        $now_end = current_time('Y-m-d');
        $now_start = current_time('Y-m-d');
        $prev_end = date('Y-m-d', strtotime('-1 day'));
        $prev_start = date('Y-m-d', strtotime('-1 day'));
        break;
      case 'WEEK':
        $now_end = current_time('Y-m-d');
        $now_start = date('Y-m-d', strtotime($now_end . '-7 days'));
        $prev_end = $now_start;
        $prev_start = date('Y-m-d', strtotime($prev_end . '-7 days'));
        break;
      case 'MONTH':
        $now_end    = current_time('Y-m-t');
        $now_start  = date('Y-m-01');
        $prev_end   = date('Y-m-d', strtotime($now_start . '-1 day'));
        $prev_start = date('Y-m-01', strtotime($now_start . '-1 month'));
        break;
      case 'YEAR':
        $now_end = current_time('Y-12-31');
        $now_start = date('Y-01-01');
        $prev_end = date('Y-12-31', strtotime($now_start . '-1 day'));
        $prev_start = date('Y-01-01', strtotime($now_start . '-1 year'));
        break;
    }

    return [
      'now_end' => $now_end,
      'now_start' => $now_start,
      'prev_end' => $prev_end,
      'prev_start' => $prev_start,
    ];
  }


  public function UltimateAiEntryRecords()
  {

    $data_days = [
      'start_date' => date('Y-m-01', strtotime('-1 month')),
      'end_date' => date('Y-m-t', strtotime('now')),
      'group_by' => 'DAY'
    ];

    $days_serial = $this->UltimateAIQueryOrders($data_days);

    $newData = [];

    foreach ($days_serial as $date) {
      $newData[$date->date][$date->package_type] = [
        'total' => (int) $date->total,
        'totalrev' => (int) $date->totalrev,
      ];
    }
    $period_range = $this->generateRangedDate($data_days['start_date'], $data_days['end_date'], 'Y-m-d');

    return array_merge($period_range, $newData);
  }
  public function UltimateAiEntryRecordsYearly()
  {

    $data_days = [
      'start_date' => date('Y-01-01', strtotime('-1 year')),
      'end_date' => date('Y-12-t', strtotime('now')),
      'group_by' => 'MONTH'
    ];

    $days_serial = $this->UltimateAIQueryOrders($data_days);

    $newData = [];

    foreach ($days_serial as $date) {
      $newData[date('M y', strtotime($date->month))][$date->package_type] = [
        'total' => (int) $date->total,
        'totalrev' => (int) $date->totalrev,
      ];
    }
    $period_range = $this->generateRangedDate($data_days['start_date'], $data_days['end_date'], 'M y');

    return array_merge($period_range, $newData);
  }

  public function generateRangedDate($start, $end, $format = 'd-m')
  {
    $period = new DatePeriod(
      new DateTime($start),
      new DateInterval('P1D'),
      new DateTime($end)
    );
    $res = [];
    foreach ($period as $key => $val) {
      $res[$val->format($format)] = 0;
    }

    return $res;
  }

  public function generationDetailsUserYear()
  {
    $user_id = get_current_user_id();
    if (!$user_id) return false;
    $data_days = [
      'start_date' => date('Y-01-01', strtotime('-1 year')),
      'end_date' => date('Y-12-t', strtotime('now')),
      'group_by' => 'MONTH',
      'author_id' => $user_id
    ];

    $days_serial = $this->UltimateAIQueryGenerationDetails($data_days);

    $newData = [];
    foreach ($days_serial as $date) {
      if ($date->tool_type == 'generate_image') {
        $newData[date('M y', strtotime($date->month))]['images'] = (int) $date->total ?? 0;
      } else {
        $newData[date('M y', strtotime($date->month))]['words'] = (int) $date->total ?? 0;
      }
    }
    $period_range = $this->generateRangedDate($data_days['start_date'], $data_days['end_date'], 'M y');

    return array_merge($period_range, $newData);
  }
  public function generationDetailsUser()
  {

    $user_id = get_current_user_id();
    if (!$user_id) return false;
    $data_days = [
      'start_date' => date('Y-m-01', strtotime('-1 month')),
      'end_date' => date('Y-m-t', strtotime('now')),
      'group_by' => 'DAY',
      'author_id' => $user_id
    ];

    $days_serial = $this->UltimateAIQueryGenerationDetails($data_days);

    $newData = [];

    foreach ($days_serial as $date) {
      if ($date->tool_type == 'generate_image') {
        $newData[$date->date]['images'] = (int) $date->total ?? 0;
      } else {
        $newData[$date->date]['words'] = (int) $date->total ?? 0;
      }
    }
    $period_range = $this->generateRangedDate($data_days['start_date'], $data_days['end_date'], 'Y-m-d');

    return array_merge($period_range, $newData);
  }
  public function generationDetailsAllUsers()
  {

    $data_days = [
      'start_date' => date('Y-m-01', strtotime('-1 month')),
      'end_date' => date('Y-m-t', strtotime('now')),
      'group_by' => 'DAY'
    ];

    $days_serial = $this->UltimateAIQueryGenerationDetails($data_days);

    $newData = [];

    foreach ($days_serial as $date) {
      if ($date->tool_type == 'generate_image') {
        $newData[$date->date]['images'] = (int) $date->total ?? 0;
      } else {
        $newData[$date->date]['words'] = (int) $date->total ?? 0;
      }
    }
    $period_range = $this->generateRangedDate($data_days['start_date'], $data_days['end_date'], 'Y-m-d');

    return array_merge($period_range, $newData);
  }
  public function generationDetailsGraphData()
  {

    $data_days = [
      'start_date' => date('Y-m-01', strtotime('-1 month')),
      'end_date' => date('Y-m-t', strtotime('now')),
      'group_by' => 'DAY'
    ];

    $days_serial = $this->UltimateAIQueryGenerationDetails($data_days);

    $newData = [];

    foreach ($days_serial as $date) {
      if ($date->tool_type == 'generate_image') {
        $newData[$date->date]['images'] = (int) $date->total ?? 0;
      } else {
        $newData[$date->date]['words'] = (int) $date->total ?? 0;
      }
    }
    $period_range = $this->generateRangedDate($data_days['start_date'], $data_days['end_date'], 'Y-m-d');

    return array_merge($period_range, $newData);
  }

  public function UltimateAiAbstractGenerationDetails()
  {

    return [
      'generated_words' => get_option('ultimate_ai_total_generation_word') ?? 0,
      'generated_images' => get_option('ultimate_ai_total_generation_image') ?? 0,
    ];
  }
  public function UltimateAIQueryOrders($data)
  {
    global $wpdb;

    $sql = "SELECT ";
    if (isset($data['group_by'])) {
      if ($data['group_by'] == 'MONTH') {
        $sql .= "MONTHNAME(p.post_date) AS month,";
      }
      if ($data['group_by'] == 'DAY') {
        $sql .= "DATE(p.post_date) AS date,";
      }
    }
    $sql .= "pkg.meta_value AS package_type,
                SUM(price.meta_value) AS totalrev,
                COUNT(price.meta_id) AS total";

    $sql .= ' FROM ' . $wpdb->prefix . 'posts AS p ';

    $sql .= 'INNER JOIN ' . $wpdb->prefix . 'postmeta AS pkg';
    $sql .= " ON p.ID = pkg.post_id AND pkg.meta_key='package_type' ";

    $sql .= 'LEFT JOIN ' . $wpdb->prefix . 'postmeta AS price';
    $sql .= " ON p.ID = price.post_id AND price.meta_key='price' ";

    $sql .= "WHERE ";

    $sql .= " pkg.meta_value IN ('subscription', 'token') ";

    if (isset($data['author_id'])) {
      $sql .= ' AND p.post_author =' . $date['author_id'];
    }
    if (isset($data['start_date']) && isset($data['end_date'])) {
      $sql .= ' AND (p.post_date BETWEEN "' . $data['start_date'] . '" AND "' . $data['end_date'] . '")';
    }
    $sql .= " GROUP BY ";
    if (isset($data['group_by'])) {
      if ($data['group_by'] == 'MONTH') {
        $sql .= "MONTH(p.post_date),";
      }
      if ($data['group_by'] == 'DAY') {
        $sql .= "DATE(p.post_date),";
      }
    }
    $sql .= " package_type";
    $res = $wpdb->get_results($sql);
    return $res;
  }
  public function UltimateAIQueryGenerationDetails($data)
  {
    global $wpdb;

    $sql = "SELECT ";
    if (isset($data['group_by'])) {
      if ($data['group_by'] == 'MONTH') {
        $sql .= "MONTHNAME(p.post_date) AS month,";
      }
      if ($data['group_by'] == 'DAY') {
        $sql .= "DATE(p.post_date) AS date,";
      }
    }
    $sql .= "tool_type.meta_value AS tool_type,
                SUM(token.meta_value) AS total";

    $sql .= ' FROM ' . $wpdb->prefix . 'posts AS p ';

    $sql .= 'INNER JOIN ' . $wpdb->prefix . 'postmeta AS tool_type';
    $sql .= " ON p.ID = tool_type.post_id AND tool_type.meta_key = 'tool_type' ";

    $sql .= 'LEFT JOIN ' . $wpdb->prefix . 'postmeta AS token';
    $sql .= " ON p.ID = token.post_id AND token.meta_key = 'token' ";

    $sql .= "WHERE ";

    $sql .= "p.post_type IN ('ultimate_ai_history')";
    if (isset($data['author_id'])) {
      $sql .= ' AND p.post_author =' . $data['author_id'];
    }
    if (isset($data['start_date']) && isset($data['end_date'])) {
      $sql .= ' AND (p.post_date BETWEEN "' . $data['start_date'] . '" AND "' . $data['end_date'] . '")';
    }
    $sql .= " GROUP BY ";
    if (isset($data['group_by'])) {
      if ($data['group_by'] == 'MONTH') {
        $sql .= "MONTH(p.post_date),";
      }
      if ($data['group_by'] == 'DAY') {
        $sql .= "DATE(p.post_date),";
      }
    }
    $sql .= " tool_type";
    $res = $wpdb->get_results($sql);
    return $res;
  }
  public function getOrderBySubscriptionID($order_id)
  {
    $args = array(
      'post_status' => 'private',
      'post_type' => 'ultimate_ai_orders',
      'fields' => 'ids',
      'meta_key' => 'order_id',
      'meta_value' => $order_id,
      'meta_compare' => '='
    );

    $id = get_posts($args);
    return $id[0];
  }
  public function deleteOrders($ids, $limit, $offset)
  {
    $res = [];
    $failed = [];
    if (is_array($ids)) {
      foreach ($ids as $id) {
        $deleted = wp_delete_post((int) $id);
        if (is_wp_error($deleted)) {
          $failed[] = $id;
        }
      }
    }
    $data = [
      'limit' => $limit,
      'offset' => $offset,
    ];
    if (empty($failed)) {
      $res = [
        'message' => esc_html__('Successfully Deleted', ULTIMATE_AI_SLUG),
        'status' => 'success',
        'orders' => $this->filterOrders($data)['all_orders']
      ];
    } else {
      $res = [
        'messsage' => sprintf(esc_html__('Deleting failed on id of : %s', ULTIMATE_AI_SLUG), implode($failed)),
        'status' => 'error',
        'orders' => $this->filterOrders($data)['all_orders']
      ];
    }
    return $res;
  }
  public function filterOrders($data = [])
  {
    $limit = isset($data['limit']) ? $data['limit'] : -1;
    $offset = isset($data['offset']) ? $data['offset'] : 0;
    $start = isset($data['start']) ? $data['start'] : '';
    $end = isset($data['end']) ? $data['end'] : '';
    $p = isset($data['order_id']) ? $data['order_id'] : '';
    $author = isset($data['author']) ? $data['author'] : '';
    $payment_method = isset($data['payment_method']) ? $data['payment_method'] : '';
    $payment_status = isset($data['payment_status']) ? $data['payment_status'] : '';

    $args = array(
      'p' => $p,
      'post_type' => 'ultimate_ai_orders',
      'post_status' => 'private',
      'fields' => 'ids',
      'posts_per_page' => intval($limit),
      'offset' => intval($offset),
      'author' => intval($author),
      'meta_query' => ["relation" => "AND"]
    );

    if (!empty($payment_status)) {
      $args['meta_query'] = [
        [
          'key' => 'payment_status',
          'value' =>  $payment_status,
          'compare' => '='
        ]
      ];
    }

    if (!empty($payment_method)) {
      $args['meta_query'] = [
        [
          'key' => 'payment_method',
          'value' =>  $payment_method,
          'compare' => '='
        ]
      ];
    }

    if (isset($data['author']) && $data['author'] > 0) {
      $args['author'] = $data['author'];
    }

    if ($start !== '' && $end !== '') {
      $args['date_query'] = array(
        array(
          'after'     => $start,
          'before'    => $end,
          'inclusive' => true,
        )
      );
    }

    // return $args;

    $orders = new WP_Query($args);
    $all_orders = [];

    if ($orders->have_posts()) {
      while ($orders->have_posts()) {
        $orders->the_post();
        $post_meta = get_post_meta(get_the_ID(), 'ultimate_ai_order_meta', true);
        if (isset($post_meta['user_id'])) {
          $user = get_user_by('id', $post_meta['user_id']);
          $post_meta['id'] = get_the_ID();
          $post_meta['plan_name'] = get_post_meta(get_the_ID(), 'package_name', true);
          // $post_meta['full_user'] = $this->USER_MANAGER->get_full_user($user);

          $all_orders[] = $post_meta;
        }
      }
    }

    return [
      'all_orders' => $all_orders,
      'total_orders' => $orders->found_posts
    ];
  }
}

